/*
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copyright (C) 1994, 95, 96, 97, 98, 99, 2003 by Ralf Baechle
 * Copyright (C) 1996 by Paul M. Antoine
 * Copyright (C) 1999 Silicon Graphics
 * Copyright (C) 2000 MIPS Technologies, Inc.
 */
#ifndef _ASM_INTERRUPT_H
#define _ASM_INTERRUPT_H

#include <linux/config.h>
#include <asm/hazards.h>

/*
 * MIPS_MT_SMTC_INSTANT_REPLAY does prompt replay of deferred IPIs,
 * at the cost of branch and call overhead on each local_irq_restore()
 */

#ifdef CONFIG_MIPS_MT_SMTC_INSTANT_REPLAY

extern void smtc_ipi_replay(void);

#define irq_restore_epilog(flags)					\
do {									\
	if (!(flags & 0x0400))						\
		smtc_ipi_replay();					\
} while (0)

#else

#define irq_restore_epilog(ignore) do { } while (0)

#endif /* CONFIG_MIPS_MT_SMTC_INSTANT_REPLAY */

__asm__ (
	"	.macro	local_irq_enable				\n"
	"	.set	push						\n"
	"	.set	reorder						\n"
	"	.set	noat						\n"
#ifdef CONFIG_CPU_MIPSR2
	"	.set	mips32r2					\n"
#ifdef CONFIG_MIPS_MT_SMTC
	"mfc0	$1, $2, 1	# SMTC - clear TCStatus.IXMT		\n"
	"ori	$1, 0x400						\n"
	"xori	$1, 0x400						\n"
	"mtc0	$1, $2, 1						\n"
#else
	"	.set	mips32r2					\n"
	"	ei							\n"
	"	.set	mips0						\n"
#endif
#else
	"	mfc0	$1,$12						\n"
	"	ori	$1,0x1f						\n"
	"	xori	$1,0x1e						\n"
	"	mtc0	$1,$12						\n"
#endif
	"	irq_enable_hazard					\n"
	"	.set	pop						\n"
	"	.endm");

static inline void local_irq_enable(void)
{
	__asm__ __volatile__(
		"local_irq_enable"
		: /* no outputs */
		: /* no inputs */
		: "memory");
}

/*
 * For cli() we have to insert nops to make sure that the new value
 * has actually arrived in the status register before the end of this
 * macro.
 * R4000/R4400 need three nops, the R4600 two nops and the R10000 needs
 * no nops at all.
 */
/*
 * For TX49, operating only IE bit is not enough.
 *
 * If mfc0 $12 follows store and the mfc0 is last instruction of a
 * page and fetching the next instruction causes TLB miss, the result
 * of the mfc0 might wrongly contain EXL bit.
 *
 * ERT-TX49H2-027, ERT-TX49H3-012, ERT-TX49HL3-006, ERT-TX49H4-008
 *
 * Workaround: mask EXL bit of the result or place a nop before mfc0.
 */
__asm__ (
	"	.macro	local_irq_disable\n"
	"	.set	push						\n"
	"	.set	noat						\n"
#ifdef CONFIG_CPU_MIPSR2
	"	.set	mips32r2					\n"
#ifdef CONFIG_MIPS_MT_SMTC
	"	mfc0	$1, $2, 1					\n"
	"	ori	$1, 0x400					\n"
	"	.set	noreorder					\n"
	"	mtc0	$1, $2, 1					\n"
#else
	"	.set	mips32r2					\n"
	"	di							\n"
	"	.set	mips0						\n"
#endif
#else
	"	mfc0	$1,$12						\n"
	"	ori	$1,0x1f						\n"
	"	xori	$1,0x1f						\n"
	"	.set	noreorder					\n"
	"	mtc0	$1,$12						\n"
#endif
	"	irq_disable_hazard					\n"
	"	.set	pop						\n"
	"	.endm							\n");

static inline void local_irq_disable(void)
{
	__asm__ __volatile__(
		"local_irq_disable"
		: /* no outputs */
		: /* no inputs */
		: "memory");
}

__asm__ (
	"	.macro	local_save_flags flags				\n"
	"	.set	push						\n"
	"	.set	reorder						\n"
#ifdef CONFIG_MIPS_MT_SMTC
	"	mfc0	\\flags, $2, 1					\n"
#else
	"	mfc0	\\flags, $12					\n"
#endif
	"	.set	pop						\n"
	"	.endm							\n");

#define local_save_flags(x)						\
__asm__ __volatile__(							\
	"local_save_flags %0"						\
	: "=r" (x))

__asm__ (
	"	.macro	local_irq_save result				\n"
	"	.set	push						\n"
	"	.set	reorder						\n"
	"	.set	noat						\n"
#ifdef CONFIG_CPU_MIPSR2
	"	.set	mips32r2					\n"
#ifdef CONFIG_MIPS_MT_SMTC
	"	mfc0	\\result, $2, 1					\n"
	"	ori	$1, \\result, 0x400				\n"
	"	.set	noreorder					\n"
	"	mtc0	$1, $2, 1					\n"
	"	andi	\\result, \\result, 0x400			\n"
#else
	"	.set	mips32r2					\n"
	"	di	\\result					\n"
	"	andi	\\result, 1					\n"
	"	.set	mips0						\n"
#endif
#else
	"	mfc0	\\result, $12					\n"
	"	ori	$1, \\result, 0x1f				\n"
	"	xori	$1, 0x1f					\n"
	"	.set	noreorder					\n"
	"	mtc0	$1, $12						\n"
#endif
	"	irq_disable_hazard					\n"
	"	.set	pop						\n"
	"	.endm							\n");

#define local_irq_save(x)						\
__asm__ __volatile__(							\
	"local_irq_save\t%0"						\
	: "=r" (x)							\
	: /* no inputs */						\
	: "memory")

__asm__ (
	"	.macro	local_irq_restore flags				\n"
	"	.set	push						\n"
	"	.set	noreorder					\n"
	"	.set	noat						\n"
#ifdef CONFIG_MIPS_MT_SMTC
	"mfc0	$1, $2, 1						\n"
	"andi	\\flags, 0x400						\n"
	"ori	$1, 0x400						\n"
	"xori	$1, 0x400						\n"
	"or	\\flags, $1						\n"
	"mtc0	\\flags, $2, 1						\n"
#elif defined(CONFIG_CPU_MIPSR2) && defined(CONFIG_IRQ_CPU)
	/*
	 * Slow, but doesn't suffer from a relativly unlikely race
	 * condition we're having since days 1.
	 */
	"	.set	mips32r2					\n"
	"	beqz	\\flags, 1f					\n"
	"	 di							\n"
	"	ei							\n"
	"	.set	mips0						\n"
	"1:								\n"
#elif defined(CONFIG_CPU_MIPSR2)
	/*
	 * Fast, dangerous.  Life is fun, life is good.
	 */
	"	mfc0	$1, $12						\n"
	"	ins	$1, \\flags, 0, 1				\n"
	"	mtc0	$1, $12						\n"
#else
	"	mfc0	$1, $12						\n"
	"	andi	\\flags, 1					\n"
	"	ori	$1, 0x1f					\n"
	"	xori	$1, 0x1f					\n"
	"	or	\\flags, $1					\n"
	"	mtc0	\\flags, $12					\n"
#endif
	"	irq_disable_hazard					\n"
	"	.set	pop						\n"
	"	.endm							\n");

#define local_irq_restore(flags)					\
do {									\
	unsigned long __tmp1;						\
									\
	__asm__ __volatile__(						\
		"local_irq_restore\t%0"					\
		: "=r" (__tmp1)						\
		: "0" (flags)						\
		: "memory");						\
	irq_restore_epilog(flags);					\
} while(0)

static inline int irqs_disabled(void)
{
#ifdef CONFIG_MIPS_MT_SMTC
	/*
	 * SMTC model uses TCStatus.IXMT to disable interrupts for a thread/CPU
	 */
	unsigned long __result;

	__asm__ __volatile__(
	"	.set	noreorder					\n"
	"	.set	mips32						\n"
	"	mfc0	%0, $2, 1					\n"
	"	andi	%0, 0x400					\n"
	"	slt	%0, $0, %0					\n"
	"	.set	mips0						\n"
	"	.set	reorder						\n"
	: "=r" (__result));

	return __result;
#else
	unsigned long flags;
	local_save_flags(flags);

	return !(flags & 1);
#endif
}

#endif /* _ASM_INTERRUPT_H */
